<?php
/**
 * Plugin Name: Spaceremit Gateway for WooCommerce
 * Plugin URI: https://spaceremit.com
 * Description: بوابة دفع Spaceremit لـ WooCommerce - تدعم الدفع بالبطاقات الائتمانية والطرق المحلية
 * Version: 1.0.0
 * Author: Spaceremit
 * Author URI: https://spaceremit.com
 * Text Domain: spaceremit-woocommerce
 * Domain Path: /languages
 * Requires at least: 5.0
 * Requires PHP: 7.2
 * WC requires at least: 3.0
 * WC tested up to: 8.0
 *
 * @package Spaceremit_WooCommerce
 */

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

// Define plugin constants.
define( 'SPACEREMIT_VERSION', '1.0.0' );
define( 'SPACEREMIT_PLUGIN_DIR', plugin_dir_path( __FILE__ ) );
define( 'SPACEREMIT_PLUGIN_URL', plugin_dir_url( __FILE__ ) );
define( 'SPACEREMIT_PLUGIN_FILE', __FILE__ );

/**
 * Lightweight bilingual translations (Arabic/English) without .mo files.
 * Keeps the gateway translatable even if language packs are missing.
 *
 * @return string Current language key (ar|en).
 */
function spaceremit_get_lang() {
	$locale = function_exists( 'determine_locale' ) ? determine_locale() : get_locale();
	return ( $locale && 0 === strpos( $locale, 'ar' ) ) ? 'ar' : 'en';
}

/**
 * Translation dictionary for both Arabic and English strings.
 *
 * @return array
 */
function spaceremit_translation_map() {
	static $translations = null;

	if ( null !== $translations ) {
		return $translations;
	}

	$translations = array(
		'en' => array(
			'Spaceremit Gateway يتطلب WooCommerce ليكون نشطاً. يرجى تفعيل WooCommerce أولاً.' => 'Spaceremit Gateway requires WooCommerce to be active. Please activate WooCommerce first.',
			'معاملات مفقودة' => 'Missing parameters.',
			'الطلب غير موجود' => 'Order not found.',
			'فشل التحقق من الدفع: %s' => 'Payment verification failed: %s',
			'تم تجاهل فرق المبلغ (الطلب: %s، الدفع مع عمولة: %s) واعتماد مبلغ الطلب.' => 'Amount difference ignored (order: %s, payment with fees: %s). Using the order total.',
			'تم التحقق من الدفع عبر Spaceremit' => 'Payment verified via Spaceremit',
			'في انتظار تأكيد الدفع من Spaceremit' => 'Awaiting payment confirmation from Spaceremit',
			'تم التحقق من الدفع من Spaceremit. Payment ID: %s, Tag: %s' => 'Payment verified from Spaceremit. Payment ID: %s, Tag: %s',
			'فشل التحقق من الدفع. الحالة: %s (Tag: %s)' => 'Payment verification failed. Status: %s (Tag: %s)',
			'تم الدفع بنجاح عبر Spaceremit' => 'Payment completed via Spaceremit',
			'تم استلام الدفع من Spaceremit. Payment ID: %s, Tag: %s' => 'Payment received from Spaceremit. Payment ID: %s, Tag: %s',
			'فشل الدفع عبر Spaceremit. الحالة: %s (Tag: %s)' => 'Payment failed via Spaceremit. Status: %s (Tag: %s)',
			'ملاحظة:' => 'Note:',
			'بعد اتمام الدفع يتم مراجعة الطلبات و قبولها خلال بضع ساعات، يرجى الانتظار.' => 'After payment, orders are reviewed and approved within a few hours. Please wait.',
			'سبيس ريمت تتيح لك استلام المدفوعات من خلال وسائل الدفع المحلية وبشكل تلقائي' => 'SpaceRemit lets you receive payments automatically through local payment methods.',
			'لديك عملية شراء جديدة قيد المراجعة، قام عميلك بارسال طلب دفع وهو بانتظار التأكيد من قبل الوكيل. الوقت المتوقع لتاكيد الطلب 0 - 12 ساعة' => 'New purchase pending review. Customer submitted a payment request awaiting agent confirmation. Estimated confirmation time: 0 - 12 hours.',
			'تم استلام دفعة و تأكيدها بنجاح، ستظهر ضمن الرصيد المتاح خلال 0 - 60 دقيقة' => 'Payment received and confirmed successfully. It will appear in the available balance within 0 - 60 minutes.',
			'لم يتمكن الوكيل من قبول عملية الدفع بسبب عدم استلام المبلغ او عدم ارفاق صورة وصل صحيحة، راجع حسابك لمعرفة السبب بالتفصيل' => 'The agent could not accept the payment because the amount was not received or the receipt image was invalid. Check your account for details.',
			'تفعيل/إلغاء' => 'Enable/Disable',
			'تفعيل بوابة دفع Spaceremit' => 'Enable Spaceremit payment gateway',
			'العنوان' => 'Title',
			'العنوان الذي يراه العميل في صفحة الدفع' => 'The title displayed to the customer on checkout.',
			'الوصف' => 'Description',
			'الوصف الذي يراه العميل في صفحة الدفع' => 'The description displayed to the customer on checkout.',
			'ادفع بواسطة المحافظ او البنوك المحلية من خلال سبيس ريمت' => 'Pay using local wallets or banks through SpaceRemit.',
			'وضع الاختبار' => 'Test mode',
			'تفعيل وضع الاختبار' => 'Enable test mode',
			'استخدم مفاتيح API الخاصة بالاختبار' => 'Use test API keys.',
			'مفتاح API العام لوضع الاختبار من Spaceremit' => 'Spaceremit test mode public API key.',
			'مفتاح API السري لوضع الاختبار من Spaceremit' => 'Spaceremit test mode secret API key.',
			'مفتاح API العام للإنتاج من Spaceremit' => 'Spaceremit live public API key.',
			'مفتاح API السري للإنتاج من Spaceremit' => 'Spaceremit live secret API key.',
			'استخدم هذا الرابط في إعدادات Spaceremit:' => 'Use this URL in your Spaceremit settings:',
			'قم بنسخ هذا الرابط وإضافته في قسم Callback URL في إعدادات موقعك على Spaceremit' => 'Copy this URL into the Callback URL section of your Spaceremit site settings.',
			'في انتظار الدفع عبر Spaceremit' => 'Awaiting payment via Spaceremit',
			'معاملات مفقودة للتحقق من الدفع' => 'Missing parameters to verify the payment.',
			'فشل التحقق من الدفع' => 'Payment verification failed.',
			'تم استلام الدفع من Spaceremit. Payment ID: ' => 'Payment received from Spaceremit. Payment ID: ',
			'تم استلام الدفع ولكن في حالة انتظار. Status: ' => 'Payment received but pending. Status: ',
			'فشل الدفع عبر Spaceremit' => 'Payment failed via Spaceremit',
			'فشل الدفع. Status: ' => 'Payment failed. Status: ',
			'تفاصيل الطلب' => 'Order details',
			'رقم الطلب:' => 'Order number:',
			'المبلغ:' => 'Amount:',
			'الدفع الآن' => 'Pay now',
			'جاري معالجة الدفع...' => 'Processing payment...',
			'فشل الدفع. يرجى المحاولة مرة أخرى.' => 'Payment failed. Please try again.',
			'حدث خطأ أثناء تحديد رابط صفحة الاستلام.' => 'An error occurred while determining the thank you page URL.',
			'حدث خطأ أثناء التحقق من الدفع.' => 'An error occurred while verifying the payment.',
			'حدث خطأ أثناء الاتصال بالخادم.' => 'An error occurred while contacting the server.',
		),
		'ar' => array(
			'Local Payments | SpaceRemit' => 'المدفوعات المحلية | سبيس ريمت',
			'Spaceremit' => 'سبيس ريمت',
			'Test Public Key' => 'مفتاح API العام لوضع الاختبار',
			'Test Secret Key' => 'مفتاح API السري لوضع الاختبار',
			'Live Public Key' => 'مفتاح API العام للإنتاج',
			'Live Secret Key' => 'مفتاح API السري للإنتاج',
			'Callback URL' => 'رابط الاستدعاء (Callback URL)',
		),
	);

	return $translations;
}

/**
 * Add an order note once per logical key to avoid duplicates.
 *
 * @param WC_Order $order   Order instance.
 * @param string   $key     Logical key for the note.
 * @param string   $message Note content.
 */
function spaceremit_add_note_once( $order, $key, $message ) {
	if ( ! $order instanceof WC_Order ) {
		return;
	}

	$logged = $order->get_meta( '_spaceremit_note_keys' );
	if ( ! is_array( $logged ) ) {
		$logged = array();
	}

	if ( in_array( $key, $logged, true ) ) {
		return;
	}

	$order->add_order_note( $message );
	$logged[] = $key;
	$order->update_meta_data( '_spaceremit_note_keys', $logged );
	$order->save_meta_data();
}

/**
 * Set order status without auto WooCommerce notes (silent transition).
 * Optionally mark as paid.
 *
 * @param WC_Order $order
 * @param string   $status
 * @param bool     $mark_paid
 */
function spaceremit_set_status_silent( $order, $status, $mark_paid = false ) {
	if ( ! $order instanceof WC_Order ) {
		return;
	}

	// Set status directly and avoid auto notes.
	$order->set_status( $status );

	if ( $mark_paid ) {
		if ( ! $order->get_date_paid() ) {
			$order->set_date_paid( time() );
		}
	}

	$order->save();
}

/**
 * Remove default WooCommerce status-change notes.
 *
 * @param WC_Order $order
 */
function spaceremit_remove_status_notes( $order ) {
	if ( ! $order instanceof WC_Order ) {
		return;
	}

	$notes = wc_get_order_notes(
		array(
			'order_id' => $order->get_id(),
			'type'     => 'internal',
		)
	);

	if ( empty( $notes ) ) {
		return;
	}

	foreach ( $notes as $note ) {
		$content = isset( $note->content ) ? $note->content : '';
		if ( stripos( $content, 'order status changed from' ) === 0 ) {
			if ( method_exists( $note, 'get_id' ) ) {
				wc_delete_order_note( $note->get_id() );
			} elseif ( isset( $note->id ) ) {
				wc_delete_order_note( $note->id );
			}
		}
	}
}

/**
 * Apply inline translations for the spaceremit text domain.
 *
 * @param string $translation Current translation.
 * @param string $text        Original text.
 * @param string $domain      Text domain.
 * @return string
 */
function spaceremit_translate_gettext( $translation, $text, $domain ) {
	if ( 'spaceremit-woocommerce' !== $domain ) {
		return $translation;
	}

	$translations = spaceremit_translation_map();
	$lang         = spaceremit_get_lang();

	if ( isset( $translations[ $lang ][ $text ] ) ) {
		return $translations[ $lang ][ $text ];
	}

	return $translation;
}
add_filter( 'gettext', 'spaceremit_translate_gettext', 10, 3 );



/**
 * Check if WooCommerce is active.
 */
function spaceremit_check_woocommerce() {
	// Better check using class_exists.
	if ( ! class_exists( 'WooCommerce' ) ) {
		add_action( 'admin_notices', 'spaceremit_woocommerce_missing_notice' );
		return false;
	}
	return true;
}

/**
 * Admin notice if WooCommerce is not active.
 */
function spaceremit_woocommerce_missing_notice() {
	?>
	<div class="error">
		<p><?php esc_html_e( 'Spaceremit Gateway يتطلب WooCommerce ليكون نشطاً. يرجى تفعيل WooCommerce أولاً.', 'spaceremit-woocommerce' ); ?></p>
	</div>
	<?php
}

/**
 * Declare WooCommerce feature compatibility.
 */
function spaceremit_declare_compatibility() {
	if ( ! class_exists( '\Automattic\WooCommerce\Utilities\FeaturesUtil' ) ) {
		if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
			error_log( 'Spaceremit Gateway: FeaturesUtil not available for compatibility declaration' );
		}
		return;
	}

	// Declare compatibility with HPOS (High-Performance Order Storage).
	$hpos_result = \Automattic\WooCommerce\Utilities\FeaturesUtil::declare_compatibility( 'custom_order_tables', SPACEREMIT_PLUGIN_FILE, true );
	if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
		error_log( 'Spaceremit Gateway: HPOS compatibility declared: ' . ( $hpos_result ? 'success' : 'failed' ) );
	}

	// Declare compatibility with Cart/Checkout Blocks.
	$blocks_result = \Automattic\WooCommerce\Utilities\FeaturesUtil::declare_compatibility( 'cart_checkout_blocks', SPACEREMIT_PLUGIN_FILE, true );
	if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
		error_log( 'Spaceremit Gateway: Blocks compatibility declared: ' . ( $blocks_result ? 'success' : 'failed' ) . ', Plugin file: ' . SPACEREMIT_PLUGIN_FILE );
	}

	// Declare also using plugin basename to avoid any mismatch.
	$plugin_basename = plugin_basename( SPACEREMIT_PLUGIN_FILE );
	$blocks_base_result = \Automattic\WooCommerce\Utilities\FeaturesUtil::declare_compatibility( 'cart_checkout_blocks', $plugin_basename, true );
	if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
		error_log( 'Spaceremit Gateway: Blocks compatibility (basename) declared: ' . ( $blocks_base_result ? 'success' : 'failed' ) . ', Plugin basename: ' . $plugin_basename );
	}
}
add_action( 'before_woocommerce_init', 'spaceremit_declare_compatibility' );

/**
 * Initialize the gateway.
 */
function spaceremit_init_gateway() {
	if ( ! spaceremit_check_woocommerce() ) {
		return;
	}

	// Load required files - use require_once to prevent duplicate loading.
	require_once __DIR__ . '/includes/class-spaceremit-gateway.php';
	require_once __DIR__ . '/includes/class-spaceremit-callback.php';

	// Register the gateway - only once via filter.
	add_filter( 'woocommerce_payment_gateways', 'spaceremit_add_gateway' );
}
add_action( 'plugins_loaded', 'spaceremit_init_gateway', 0 );

/**
 * Load Blocks Integration class early when Blocks are available.
 * This must be loaded before woocommerce_blocks_loaded hook.
 */
function spaceremit_load_blocks_integration() {
	// Prevent duplicate loading.
	if ( class_exists( 'Spaceremit_Blocks_Integration' ) ) {
		return;
	}

	if ( ! class_exists( '\Automattic\WooCommerce\Blocks\Payments\Integrations\AbstractPaymentMethodType' ) ) {
		if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
			error_log( 'Spaceremit Gateway: Blocks API not available yet' );
		}
		return;
	}
	
	if ( ! file_exists( __DIR__ . '/includes/class-spaceremit-blocks-integration.php' ) ) {
		if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
			error_log( 'Spaceremit Gateway: Integration class file not found' );
		}
		return;
	}
	
	require_once __DIR__ . '/includes/class-spaceremit-blocks-integration.php';
	
	if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
		error_log( 'Spaceremit Gateway: Blocks Integration class loaded' );
	}
}
// Load early - before woocommerce_blocks_loaded (which fires at priority 10)
add_action( 'woocommerce_blocks_loaded', 'spaceremit_load_blocks_integration', 5 );

/**
 * Add the gateway to WooCommerce.
 *
 * @param array $gateways Existing gateways.
 * @return array
 */
function spaceremit_add_gateway( $gateways ) {
	if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
		error_log( 'Spaceremit Gateway: spaceremit_add_gateway() called. Adding gateway.' );
	}
	$gateways[] = 'WC_Spaceremit_Gateway';
	if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
		error_log( 'Spaceremit Gateway: Gateway added. Total gateways: ' . count( $gateways ) );
	}
	return $gateways;
}

/**
 * Load plugin textdomain.
 */
function spaceremit_load_textdomain() {
	load_plugin_textdomain( 'spaceremit-woocommerce', false, dirname( plugin_basename( __FILE__ ) ) . '/languages' );
}
add_action( 'plugins_loaded', 'spaceremit_load_textdomain' );

/**
 * Debug hook to check available gateways - with higher priority to run AFTER other filters.
 */
function spaceremit_debug_available_gateways( $gateways ) {
	if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
		$is_checkout = is_checkout() || is_checkout_pay_page() || ( defined( 'REST_REQUEST' ) && REST_REQUEST );
		if ( $is_checkout ) {
			error_log( 'Spaceremit Gateway: Available gateways filter called. Count: ' . count( $gateways ) );
			$gateway_ids = array_keys( $gateways );
			error_log( 'Spaceremit Gateway: Available gateway IDs: ' . implode( ', ', $gateway_ids ) );
			if ( isset( $gateways['spaceremit'] ) ) {
				error_log( 'Spaceremit Gateway: Spaceremit gateway IS in available gateways! Enabled: ' . $gateways['spaceremit']->enabled );
			} else {
				error_log( 'Spaceremit Gateway: Spaceremit gateway is NOT in available gateways!' );
				// Check if it exists but was filtered out
				if ( function_exists( 'WC' ) && WC()->payment_gateways ) {
					$all_gateways = WC()->payment_gateways->payment_gateways();
					if ( isset( $all_gateways['spaceremit'] ) ) {
						$spaceremit = $all_gateways['spaceremit'];
						error_log( 'Spaceremit Gateway: Gateway exists but not available. Enabled: ' . $spaceremit->enabled . ', is_available: ' . ( $spaceremit->is_available() ? 'true' : 'false' ) );
					}
				}
			}
		}
	}
	return $gateways;
}
add_filter( 'woocommerce_available_payment_gateways', 'spaceremit_debug_available_gateways', 9999 );

/**
 * Ensure spaceremit gateway is always included if it's available.
 * This runs with highest priority to ensure gateway is included even if other filters remove it
 */
function spaceremit_ensure_gateway_available( $gateways ) {
	// Always check, not just on checkout pages (for Store API)
	if ( ! is_array( $gateways ) ) {
		return $gateways;
	}

	// Check if spaceremit gateway exists and is available but was removed by another filter
	if ( ! isset( $gateways['spaceremit'] ) ) {
		$gateway_class = 'WC_Spaceremit_Gateway';
		if ( class_exists( $gateway_class ) ) {
			$gateway_instance = new $gateway_class();
			if ( $gateway_instance->is_available() ) {
				$gateways['spaceremit'] = $gateway_instance;
				if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
					error_log( 'Spaceremit Gateway: Gateway was removed by another filter, restoring it! Enabled: ' . $gateway_instance->enabled );
				}
			} else {
				if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
					error_log( 'Spaceremit Gateway: Gateway is not available. Enabled: ' . $gateway_instance->enabled );
				}
			}
		}
	} else {
		// Gateway is in the list, ensure enabled is correct
		if ( is_a( $gateways['spaceremit'], 'WC_Spaceremit_Gateway' ) ) {
			$enabled_option = $gateways['spaceremit']->get_option( 'enabled' );
			$gateways['spaceremit']->enabled = ( ! empty( $enabled_option ) && 'yes' === $enabled_option ) ? 'yes' : 'no';
		}
	}
	
	return $gateways;
}
add_filter( 'woocommerce_available_payment_gateways', 'spaceremit_ensure_gateway_available', 99999 );

/**
 * Initialize global flag to track if Blocks Integration has been registered.
 * This prevents duplicate registration when constructor is called multiple times.
 */
if ( ! isset( $GLOBALS['spaceremit_blocks_registered'] ) ) {
	$GLOBALS['spaceremit_blocks_registered'] = false;
}

/**
 * Register Spaceremit gateway in Blocks Payment Method Registry.
 * This function is called once when Blocks are loaded.
 */
function spaceremit_register_gateway( $payment_method_registry ) {
	// Prevent duplicate registration.
	if ( isset( $GLOBALS['spaceremit_blocks_registered'] ) && $GLOBALS['spaceremit_blocks_registered'] ) {
		return;
	}

	if ( ! class_exists( 'Spaceremit_Blocks_Integration' ) ) {
		if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
			error_log( 'Spaceremit Gateway: Integration class not available during registration' );
		}
		return;
	}
	
	$integration = new Spaceremit_Blocks_Integration();
	$integration->initialize(); // Initialize before registration (like PayPal does)
	
	// Check if integration is active before registering
	if ( ! $integration->is_active() ) {
		if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
			error_log( 'Spaceremit Gateway: Integration is not active (gateway may be disabled)' );
		}
		return;
	}
	
	$payment_method_registry->register( $integration );
	
	// Mark as registered to prevent duplicates.
	$GLOBALS['spaceremit_blocks_registered'] = true;
	
	if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
		error_log( 'Spaceremit Gateway: Blocks Integration registered successfully!' );
	}
}

/**
 * Register Spaceremit Blocks Integration.
 * This makes the gateway compatible with WooCommerce Blocks Checkout.
 * Must run on woocommerce_blocks_loaded hook (like PayPal does).
 */
function spaceremit_register_blocks_integration() {
	// Prevent duplicate registration attempts.
	if ( isset( $GLOBALS['spaceremit_blocks_registered'] ) && $GLOBALS['spaceremit_blocks_registered'] ) {
		return;
	}

	// Check if Blocks API is available
	if ( ! class_exists( '\Automattic\WooCommerce\Blocks\Payments\Integrations\AbstractPaymentMethodType' ) ) {
		if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
			error_log( 'Spaceremit Gateway: Cannot register - Blocks API not available' );
		}
		return;
	}

	// Check if our Integration class is loaded
	if ( ! class_exists( 'Spaceremit_Blocks_Integration' ) ) {
		// Try to load it if not already loaded
		if ( file_exists( __DIR__ . '/includes/class-spaceremit-blocks-integration.php' ) ) {
			require_once __DIR__ . '/includes/class-spaceremit-blocks-integration.php';
		}
		
		if ( ! class_exists( 'Spaceremit_Blocks_Integration' ) ) {
			if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
				error_log( 'Spaceremit Gateway: Cannot register - Integration class not found' );
			}
			return;
		}
	}

	if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
		error_log( 'Spaceremit Gateway: Registering Blocks Integration on woocommerce_blocks_payment_method_type_registration hook' );
	}

	// Register the integration on payment method type registration hook - check if already registered.
	if ( ! has_action( 'woocommerce_blocks_payment_method_type_registration', 'spaceremit_register_gateway' ) ) {
		add_action( 'woocommerce_blocks_payment_method_type_registration', 'spaceremit_register_gateway', 10, 1 );
	}
}
// Register on woocommerce_blocks_loaded hook (like PayPal's BlocksModule)
add_action( 'woocommerce_blocks_loaded', 'spaceremit_register_blocks_integration', 10 );

/**
 * Debug: Check compatibility status when Block Editor loads.
 * This helps diagnose why the warning message appears.
 */
function spaceremit_debug_blocks_compatibility() {
	if ( ! defined( 'WP_DEBUG' ) || ! WP_DEBUG ) {
		return;
	}
	
	// Run earlier to catch Block Editor loading
	if ( ! function_exists( 'get_current_screen' ) ) {
		return;
	}
	
	$screen = get_current_screen();
	if ( ! $screen || ! $screen->is_block_editor() ) {
		return;
	}
	
	// Check if FeaturesUtil is available
	if ( ! class_exists( '\Automattic\WooCommerce\Utilities\FeaturesUtil' ) ) {
		error_log( 'Spaceremit Gateway: FeaturesUtil not available in Block Editor' );
		return;
	}
	
	// Check compatibility status - try multiple plugin ID formats
	$plugin_file = plugin_basename( SPACEREMIT_PLUGIN_FILE );
	$plugin_data = get_plugin_data( SPACEREMIT_PLUGIN_FILE, false, false );
	$text_domain = $plugin_data['TextDomain'] ?? '';
	
	error_log( 'Spaceremit Gateway: === BLOCK EDITOR COMPATIBILITY CHECK ===' );
	error_log( 'Spaceremit Gateway: Plugin file: ' . SPACEREMIT_PLUGIN_FILE );
	error_log( 'Spaceremit Gateway: Plugin basename: ' . $plugin_file );
	error_log( 'Spaceremit Gateway: TextDomain: ' . $text_domain );
	
	// Check from plugin perspective
	$compatibility = \Automattic\WooCommerce\Utilities\FeaturesUtil::get_compatible_features_for_plugin( $plugin_file );
	error_log( 'Spaceremit Gateway: Compatibility status for plugin "' . $plugin_file . "':" );
	error_log( '  - Compatible features: ' . implode( ', ', $compatibility['compatible'] ?? array() ) );
	error_log( '  - Incompatible features: ' . implode( ', ', $compatibility['incompatible'] ?? array() ) );
	error_log( '  - Uncertain features: ' . implode( ', ', $compatibility['uncertain'] ?? array() ) );
	error_log( '  - cart_checkout_blocks compatible: ' . ( in_array( 'cart_checkout_blocks', $compatibility['compatible'] ?? array(), true ) ? 'YES' : 'NO' ) );
	error_log( '  - cart_checkout_blocks incompatible: ' . ( in_array( 'cart_checkout_blocks', $compatibility['incompatible'] ?? array(), true ) ? 'YES' : 'NO' ) );
	
	// Also check from feature perspective
	$plugins_for_feature = \Automattic\WooCommerce\Utilities\FeaturesUtil::get_compatible_plugins_for_feature( 'cart_checkout_blocks', true, false );
	error_log( 'Spaceremit Gateway: Plugins for cart_checkout_blocks feature (active only, no resolve):' );
	error_log( '  - Compatible plugins: ' . implode( ', ', $plugins_for_feature['compatible'] ?? array() ) );
	error_log( '  - Incompatible plugins: ' . implode( ', ', $plugins_for_feature['incompatible'] ?? array() ) );
	error_log( '  - Uncertain plugins: ' . implode( ', ', $plugins_for_feature['uncertain'] ?? array() ) );
	error_log( '  - Plugin "' . $plugin_file . '" in incompatible list: ' . ( in_array( $plugin_file, $plugins_for_feature['incompatible'] ?? array(), true ) ? 'YES âš ï¸' : 'NO âœ…' ) );
	error_log( '  - Plugin "' . $plugin_file . '" in compatible list: ' . ( in_array( $plugin_file, $plugins_for_feature['compatible'] ?? array(), true ) ? 'YES âœ…' : 'NO' ) );
	
	// Check what TextDomain will be used for display
	$all_plugins = get_plugins();
	if ( isset( $all_plugins[ $plugin_file ] ) ) {
		$plugin_info = $all_plugins[ $plugin_file ];
		$display_id = $plugin_info['TextDomain'] ?? dirname( $plugin_file, 2 );
		error_log( 'Spaceremit Gateway: Display ID (TextDomain or dirname): ' . $display_id );
		error_log( 'Spaceremit Gateway: Plugin Name: ' . ( $plugin_info['Name'] ?? 'N/A' ) );
	}
	error_log( 'Spaceremit Gateway: === END COMPATIBILITY CHECK ===' );
}
// Run early to catch Block Editor - use multiple hooks to ensure we catch it
add_action( 'current_screen', 'spaceremit_debug_blocks_compatibility', 10 );
add_action( 'admin_init', 'spaceremit_debug_blocks_compatibility', 999 );
add_action( 'enqueue_block_editor_assets', 'spaceremit_debug_blocks_compatibility', 1 );

/**
 * Fallback: Try to register Integration if Blocks are loaded later.
 * This ensures registration even if woocommerce_blocks_loaded fired before our code.
 */
function spaceremit_register_blocks_integration_fallback() {
	// Prevent duplicate registration.
	if ( isset( $GLOBALS['spaceremit_blocks_registered'] ) && $GLOBALS['spaceremit_blocks_registered'] ) {
		return;
	}

	// Only run if Blocks are already loaded and we haven't registered yet
	if ( ! did_action( 'woocommerce_blocks_payment_method_type_registration' ) ) {
		return;
	}
	
	// Check if already registered by checking if hook was fired
	if ( ! class_exists( '\Automattic\WooCommerce\Blocks\Payments\Integrations\AbstractPaymentMethodType' ) ) {
		return;
	}
	
	if ( ! class_exists( 'Spaceremit_Blocks_Integration' ) ) {
		if ( file_exists( __DIR__ . '/includes/class-spaceremit-blocks-integration.php' ) ) {
			require_once __DIR__ . '/includes/class-spaceremit-blocks-integration.php';
		}
	}
	
	if ( class_exists( 'Spaceremit_Blocks_Integration' ) ) {
		// Try to get the registry and register directly
		// This is a fallback if normal registration didn't work
		if ( function_exists( 'woocommerce_store_api_register_payment_requirements' ) ) {
			if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
				error_log( 'Spaceremit Gateway: Fallback registration attempted (Blocks already initialized)' );
			}
		}
	}
}
// Try fallback after init (lower priority)
add_action( 'init', 'spaceremit_register_blocks_integration_fallback', 20 );


/**
 * Handle AJAX payment verification.
 */
function spaceremit_ajax_verify_payment() {
	check_ajax_referer( 'spaceremit_payment_nonce', 'nonce' );

	$payment_code = isset( $_POST['payment_code'] ) ? sanitize_text_field( wp_unslash( $_POST['payment_code'] ) ) : '';
	$order_id     = isset( $_POST['order_id'] ) ? absint( $_POST['order_id'] ) : 0;

	if ( empty( $payment_code ) || empty( $order_id ) ) {
		wp_send_json_error( array( 'message' => __( 'معاملات مفقودة', 'spaceremit-woocommerce' ) ) );
	}

	$order = wc_get_order( $order_id );

	if ( ! $order ) {
		wp_send_json_error( array( 'message' => __( 'الطلب غير موجود', 'spaceremit-woocommerce' ) ) );
	}

	$gateway = new WC_Spaceremit_Gateway();

	// Prevent duplicate processing
$processed_flag = $order->get_meta( '_spaceremit_processed' );
if ( 'yes' === $processed_flag ) {
	wp_send_json_success( array( 'payment_id' => $order->get_meta( '_spaceremit_payment_id' ) ) );
	exit;
}
if ( 'processing' === $processed_flag ) {
	wp_send_json_success( array( 'payment_id' => $order->get_meta( '_spaceremit_payment_id' ) ) );
	exit;
}
$order->update_meta_data( '_spaceremit_processed', 'processing' );
$order->save_meta_data();

	$verification = $gateway->verify_payment( $payment_code );

	if ( is_wp_error( $verification ) ) {
		wp_send_json_error( array( 'message' => $verification->get_error_message() ) );
	}

	$payment_data = $verification['data'];
	$payment_id = $payment_data['id'];

	// Store payment ID in order meta.
	$order->update_meta_data( '_spaceremit_payment_id', $payment_id );
	$order->update_meta_data( '_spaceremit_payment_code', $payment_code );

		// Normalize amount: if the gateway returns an amount with fees, rely on the order total and ignore the difference.
		$order_amount   = (float) $order->get_total();
		$payment_amount = isset( $payment_data['total_amount'] ) ? (float) $payment_data['total_amount'] : $order_amount;
		$payment_data['total_amount'] = $order_amount; // Force using the order total for verification.

		// Ignore differences silently; rely on order total.

	// Update order status based on payment status.
	$status_tag = $payment_data['status_tag'];

	// Always keep order Pending for manual review.
	spaceremit_set_status_silent( $order, 'pending', false );
	spaceremit_add_note_once(
		$order,
		'spaceremit_pending',
		__( 'لديك عملية شراء جديدة قيد المراجعة، قام عميلك بارسال طلب دفع وهو بانتظار التأكيد من قبل الوكيل. الوقت المتوقع لتاكيد الطلب 0 - 12 ساعة', 'spaceremit-woocommerce' )
	);

	// Clean default WC status-change notes
	spaceremit_remove_status_notes( $order );

	$order->update_meta_data( '_spaceremit_processed', 'yes' );
	$order->save();

	wp_send_json_success( array( 'payment_id' => $payment_id ) );
}
add_action( 'wp_ajax_spaceremit_verify_payment', 'spaceremit_ajax_verify_payment' );
add_action( 'wp_ajax_nopriv_spaceremit_verify_payment', 'spaceremit_ajax_verify_payment' );

/**
 * Catch return from Spaceremit when redirected back with SP_payment_code.
 * Verify server-side then redirect to order-received without extra params.
 */
function spaceremit_catch_return_redirect() {
	if ( is_admin() || wp_doing_ajax() ) {
		return;
	}

	if ( ! isset( $_GET['SP_payment_code'] ) ) {
		return;
	}

	$payment_code = sanitize_text_field( wp_unslash( $_GET['SP_payment_code'] ) );
	
	// Get WooCommerce endpoint settings
	$order_pay_endpoint = get_option( 'woocommerce_checkout_pay_endpoint', 'order-pay' );
	
	// Try to get order_id from URL path first (using actual WooCommerce endpoint)
	$order_id = 0;
	if ( isset( $_SERVER['REQUEST_URI'] ) ) {
		$request_uri = sanitize_text_field( wp_unslash( $_SERVER['REQUEST_URI'] ) );
		// Match {order-pay-endpoint}/{order_id}/ pattern using actual endpoint
		$pattern = '/' . preg_quote( $order_pay_endpoint, '/' ) . '\/(\d+)/';
		if ( preg_match( $pattern, $request_uri, $matches ) ) {
			$order_id = absint( $matches[1] );
		}
	}
	
	// Fallback to GET parameter if not found in path
	if ( ! $order_id && isset( $_GET['order'] ) ) {
		$order_id = absint( $_GET['order'] );
	}
	
	// Also try to extract from query var if WooCommerce sets it (using actual endpoint name)
	if ( ! $order_id ) {
		$order_pay_id = get_query_var( $order_pay_endpoint );
		if ( $order_pay_id ) {
			$order_id = absint( $order_pay_id );
		}
	}
	
	// CRITICAL: Spaceremit returns to home page with notes parameter containing order number
	// Extract order_id from notes parameter (format: notes=94 means order number 94)
	if ( ! $order_id && isset( $_GET['notes'] ) ) {
		$notes = sanitize_text_field( wp_unslash( $_GET['notes'] ) );
		
		if ( is_numeric( $notes ) ) {
			// Try direct order ID first (most common case)
			$test_order = wc_get_order( absint( $notes ) );
			if ( $test_order ) {
				$order_id = absint( $notes );
			} else {
				// Try to find by order number (if using sequential order numbers plugin)
				global $wpdb;
				$meta_table = $wpdb->prefix . 'wc_orders_meta';
				// Try HPOS meta table first
				if ( $wpdb->get_var( $wpdb->prepare( "SHOW TABLES LIKE %s", $meta_table ) ) === $meta_table ) {
					$order_ids = $wpdb->get_col(
						$wpdb->prepare(
							"SELECT DISTINCT order_id FROM {$meta_table}
							WHERE meta_key = %s AND meta_value = %s
							LIMIT 1",
							'_order_number',
							$notes
						)
					);
					if ( ! empty( $order_ids ) ) {
						$order_id = absint( $order_ids[0] );
					}
				}
				// Fallback to postmeta
				if ( ! $order_id ) {
					$order_ids = $wpdb->get_col(
						$wpdb->prepare(
							"SELECT DISTINCT post_id FROM {$wpdb->postmeta}
							WHERE meta_key = %s AND meta_value = %s
							LIMIT 1",
							'_order_number',
							$notes
						)
					);
					if ( ! empty( $order_ids ) ) {
						$order_id = absint( $order_ids[0] );
					}
				}
				// Last resort: try by order number using WooCommerce function if available
				if ( ! $order_id && function_exists( 'wc_get_order_id_by_order_key' ) ) {
					// This won't work directly, but we can try searching orders
					$orders = wc_get_orders(
						array(
							'limit'      => 1,
							'meta_key'   => '_order_number',
							'meta_value' => $notes,
						)
					);
					if ( ! empty( $orders ) ) {
						$order_id = $orders[0]->get_id();
					}
				}
			}
		}
	}
	
	// Last resort: search by payment code in order meta (if already stored)
	if ( ! $order_id && $payment_code ) {
		global $wpdb;
		$meta_table = $wpdb->prefix . 'wc_orders_meta';
		// Try HPOS meta table first
		if ( $wpdb->get_var( $wpdb->prepare( "SHOW TABLES LIKE %s", $meta_table ) ) === $meta_table ) {
			$order_ids = $wpdb->get_col(
				$wpdb->prepare(
					"SELECT DISTINCT order_id FROM {$meta_table}
					WHERE meta_key = %s AND meta_value = %s
					LIMIT 1",
					'_spaceremit_payment_code',
					$payment_code
				)
			);
			if ( ! empty( $order_ids ) ) {
				$order_id = absint( $order_ids[0] );
			}
		}
		// Fallback to postmeta for legacy
		if ( ! $order_id ) {
			$order_ids = $wpdb->get_col(
				$wpdb->prepare(
					"SELECT DISTINCT post_id FROM {$wpdb->postmeta}
					WHERE meta_key = %s AND meta_value = %s
					LIMIT 1",
					'_spaceremit_payment_code',
					$payment_code
				)
			);
			if ( ! empty( $order_ids ) ) {
				$order_id = absint( $order_ids[0] );
			}
		}
	}

	if ( ! $order_id || ! $payment_code ) {
		return;
	}

	$order = wc_get_order( $order_id );
	if ( ! $order ) {
		return;
	}

// Prevent duplicate processing (lock)
$processed_flag = $order->get_meta( '_spaceremit_processed' );
if ( 'yes' === $processed_flag ) {
	$thank_you_url = $order->get_checkout_order_received_url();
	wp_safe_redirect( $thank_you_url );
	exit;
}
if ( 'processing' === $processed_flag ) {
	$thank_you_url = $order->get_checkout_order_received_url();
	wp_safe_redirect( $thank_you_url );
	exit;
}
$order->update_meta_data( '_spaceremit_processed', 'processing' );
$order->save_meta_data();

	$gateway      = new WC_Spaceremit_Gateway();
	$verification = $gateway->verify_payment( $payment_code );

	if ( is_wp_error( $verification ) ) {
		wc_add_notice( $verification->get_error_message(), 'error' );
		wp_safe_redirect( $order->get_checkout_payment_url( true ) );
		exit;
	}

	$payment_data = $verification['data'];
	$payment_id   = $payment_data['id'] ?? '';
	$status_tag   = $payment_data['status_tag'] ?? '';
	$status       = $payment_data['status'] ?? '';

	$order->update_meta_data( '_spaceremit_payment_id', $payment_id );
	$order->update_meta_data( '_spaceremit_payment_code', $payment_code );
	$order->update_meta_data( '_spaceremit_payment_status', $status );
	$order->update_meta_data( '_spaceremit_payment_status_tag', $status_tag );

	$order_amount   = (float) $order->get_total();
	$payment_amount = isset( $payment_data['total_amount'] ) ? (float) $payment_data['total_amount'] : $order_amount;
	$payment_data['total_amount'] = $order_amount;
	if ( abs( $order_amount - $payment_amount ) > 0.01 ) {
		// Ignore silently; enforce order total.
	}

	// Always keep order Pending for manual review.
	spaceremit_set_status_silent( $order, 'pending', false );
	spaceremit_add_note_once(
		$order,
		'spaceremit_pending',
		__( 'لديك عملية شراء جديدة قيد المراجعة، قام عميلك بارسال طلب دفع وهو بانتظار التأكيد من قبل الوكيل. الوقت المتوقع لتاكيد الطلب 0 - 12 ساعة', 'spaceremit-woocommerce' )
	);

	// Clean default WC status-change notes
	spaceremit_remove_status_notes( $order );

	$order->update_meta_data( '_spaceremit_processed', 'yes' );
	$order->save();
	
	// Get the order received URL using WooCommerce's method (respects all settings)
	$thank_you_url = $order->get_checkout_order_received_url();
	
	// Ensure we have a valid URL
	if ( empty( $thank_you_url ) ) {
		// Fallback: construct manually using WooCommerce settings
		$checkout_url = wc_get_checkout_url();
		$order_received_endpoint = get_option( 'woocommerce_checkout_order_received_endpoint', 'order-received' );
		$thank_you_url = wc_get_endpoint_url( $order_received_endpoint, $order_id, $checkout_url );
		$thank_you_url = add_query_arg( 'key', $order->get_order_key(), $thank_you_url );
	}
	
	wp_safe_redirect( $thank_you_url );
	exit;
}
add_action( 'template_redirect', 'spaceremit_catch_return_redirect', 1 );

/**
 * Add custom message on thank you page after Spaceremit payment.
 *
 * @param int $order_id Order ID.
 */
function spaceremit_thankyou_message( $order_id ) {
	if ( ! $order_id ) {
		return;
	}

	$order = wc_get_order( $order_id );
	if ( ! $order || 'spaceremit' !== $order->get_payment_method() ) {
		return;
	}

	?>
	<div class="spaceremit-thankyou-message" style="margin: 20px 0; padding: 15px; background-color: #f0f0f0; border-left: 4px solid #0073aa; border-radius: 4px;">
		<p style="margin: 0; font-size: 14px; line-height: 1.6;">
			<strong><?php esc_html_e( 'ملاحظة:', 'spaceremit-woocommerce' ); ?></strong><br>
			<?php esc_html_e( 'بعد اتمام الدفع يتم مراجعة الطلبات و قبولها خلال بضع ساعات، يرجى الانتظار.', 'spaceremit-woocommerce' ); ?>
		</p>
	</div>
	<?php
}
add_action( 'woocommerce_thankyou_spaceremit', 'spaceremit_thankyou_message', 10, 1 );

