<?php
/**
 * Spaceremit Payment Gateway for WooCommerce
 *
 * @package Spaceremit_WooCommerce
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * WC_Spaceremit_Gateway Class.
 */
class WC_Spaceremit_Gateway extends WC_Payment_Gateway {

	/**
	 * Whether test mode is enabled.
	 *
	 * @var bool
	 */
	public $testmode;

	/**
	 * Public key for API authentication.
	 *
	 * @var string
	 */
	public $public_key;

	/**
	 * Secret key for API authentication.
	 *
	 * @var string
	 */
	public $secret_key;

	/**
	 * Whether local payment methods are enabled.
	 *
	 * @var bool
	 */
	public $local_methods;


	/**
	 * Constructor for the gateway.
	 */
	public function __construct() {
		$this->id                 = 'spaceremit';
		$this->icon               = apply_filters( 'spaceremit_gateway_icon', '' ); // Allow filtering icon
		$this->has_fields         = false;
		$this->method_title       = __( 'Local Payments | SpaceRemit', 'spaceremit-woocommerce' );
		$this->method_description = __( 'سبيس ريمت تتيح لك استلام المدفوعات من خلال وسائل الدفع المحلية وبشكل تلقائي', 'spaceremit-woocommerce' );
		$this->supports           = array(
			'products',
			'payment_form', // Support for payment form fields if needed
		);
		
		// Set max_amount to 0 (no limit) to prevent parent::is_available() from blocking the gateway.
		$this->max_amount         = 0;

		// Load the settings.
		$this->init_form_fields();
		$this->init_settings(); // This sets $this->enabled from $this->settings['enabled']

		// Define user set variables.
		$this->title              = $this->get_option( 'title' );
		$this->description        = $this->get_option( 'description' );
		$this->testmode           = 'yes' === $this->get_option( 'testmode', 'yes' );
		$this->public_key         = $this->testmode ? $this->get_option( 'test_public_key' ) : $this->get_option( 'public_key' );
		$this->secret_key         = $this->testmode ? $this->get_option( 'test_secret_key' ) : $this->get_option( 'secret_key' );
		
		// CRITICAL: Ensure enabled is set correctly from settings (required for Blocks compatibility)
		// parent::init_settings() should set this, but we ensure it's correct
		// Use the value from settings array (which init_settings() loaded) to be consistent
		$enabled_option = isset( $this->settings['enabled'] ) ? $this->settings['enabled'] : $this->get_option( 'enabled' );
		$this->enabled  = ( ! empty( $enabled_option ) && 'yes' === $enabled_option ) ? 'yes' : 'no';
		
		// CRITICAL: Also update settings array to ensure consistency
		$this->settings['enabled'] = $this->enabled;
		
		// Local methods is always enabled (required)
		$this->local_methods = true;

		// Debug logging
		if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
			error_log( 'Spaceremit Gateway: Constructor called. Enabled: ' . $this->enabled . ', Testmode: ' . ( $this->testmode ? 'yes' : 'no' ) );
		}

		// Actions.
		add_action( 'woocommerce_update_options_payment_gateways_' . $this->id, array( $this, 'process_admin_options' ) );
		add_action( 'woocommerce_receipt_' . $this->id, array( $this, 'receipt_page' ) );
		add_action( 'wp_enqueue_scripts', array( $this, 'enqueue_scripts' ) );
	}

	/**
	 * Check if this gateway is available for use.
	 *
	 * @return bool
	 */
	public function is_available() {
		// Always check fresh settings from database.
		$enabled = $this->get_option( 'enabled' );
		$testmode = 'yes' === $this->get_option( 'testmode', 'yes' );
		$public_key = $testmode ? $this->get_option( 'test_public_key' ) : $this->get_option( 'public_key' );
		$secret_key = $testmode ? $this->get_option( 'test_secret_key' ) : $this->get_option( 'secret_key' );

		// Debug logging
		if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
			error_log( 'Spaceremit Gateway: is_available() called. Enabled: ' . $enabled . ', Testmode: ' . ( $testmode ? 'yes' : 'no' ) );
		}

		// Check if gateway is enabled (must be exactly 'yes').
		if ( 'yes' !== $enabled ) {
			if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
				error_log( 'Spaceremit Gateway: Gateway not enabled. Enabled value: ' . $enabled );
			}
			return false;
		}

		// Check if public key is set (required).
		if ( empty( $public_key ) ) {
			if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
				error_log( 'Spaceremit Gateway: Public key is empty. Test mode: ' . ( $testmode ? 'yes' : 'no' ) );
			}
			return false;
		}

		// Check if secret key is set (required).
		if ( empty( $secret_key ) ) {
			if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
				error_log( 'Spaceremit Gateway: Secret key is empty. Test mode: ' . ( $testmode ? 'yes' : 'no' ) );
			}
			return false;
		}

		// Update instance variables so parent::is_available() can use them.
		// This must be done AFTER our checks but BEFORE calling parent::is_available().
		$this->enabled = 'yes'; // Set to 'yes' since we already verified it above.
		$this->testmode = $testmode;
		$this->public_key = $public_key;
		$this->secret_key = $secret_key;

		// All our checks passed. Now check parent (which checks enabled status, cart availability, and other WooCommerce conditions).
		// Parent will check if gateway is enabled (we already did, but parent needs $this->enabled to be set to 'yes').
		// Note: We don't check cart here because parent::is_available() handles cart availability checks.
		$parent_available = parent::is_available();
		
		// Debug logging
		if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
			error_log( 'Spaceremit Gateway: parent::is_available() returned: ' . ( $parent_available ? 'true' : 'false' ) . ', $this->enabled: ' . $this->enabled );
			if ( ! $parent_available ) {
				error_log( 'Spaceremit Gateway: parent::is_available() returned false. Enabled: ' . $this->enabled . ', max_amount: ' . $this->max_amount );
				if ( WC()->cart ) {
					error_log( 'Spaceremit Gateway: Cart exists, total: ' . WC()->cart->get_total( 'edit' ) );
				} else {
					error_log( 'Spaceremit Gateway: Cart is null or not available' );
				}
			}
		}

		return $parent_available;
	}

	/**
	 * Process admin options and save settings.
	 */
	public function process_admin_options() {
		parent::process_admin_options();

		// Re-initialize settings after save.
		$this->init_settings();

		// Re-define variables after save.
		$this->title              = $this->get_option( 'title' );
		$this->description        = $this->get_option( 'description' );
		$this->testmode           = 'yes' === $this->get_option( 'testmode', 'yes' );
		$this->public_key         = $this->testmode ? $this->get_option( 'test_public_key' ) : $this->get_option( 'public_key' );
		$this->secret_key         = $this->testmode ? $this->get_option( 'test_secret_key' ) : $this->get_option( 'secret_key' );
		$this->enabled            = $this->get_option( 'enabled' );
		// Local methods is always enabled (required)
		$this->local_methods = true;
	}

	/**
	 * Initialize Gateway Settings Form Fields.
	 */
	public function init_form_fields() {
		$this->form_fields = array(
			'enabled'            => array(
				'title'       => __( 'تفعيل/إلغاء', 'spaceremit-woocommerce' ),
				'label'       => __( 'تفعيل بوابة دفع Spaceremit', 'spaceremit-woocommerce' ),
				'type'        => 'checkbox',
				'description' => '',
				'default'     => 'no',
			),
			'title'              => array(
				'title'       => __( 'العنوان', 'spaceremit-woocommerce' ),
				'type'        => 'text',
				'description' => __( 'العنوان الذي يراه العميل في صفحة الدفع', 'spaceremit-woocommerce' ),
				'default'     => __( 'Local Payments | SpaceRemit', 'spaceremit-woocommerce' ),
				'desc_tip'    => true,
			),
			'description'        => array(
				'title'       => __( 'الوصف', 'spaceremit-woocommerce' ),
				'type'        => 'textarea',
				'description' => __( 'الوصف الذي يراه العميل في صفحة الدفع', 'spaceremit-woocommerce' ),
				'default'     => __( 'ادفع بواسطة المحافظ او البنوك المحلية من خلال سبيس ريمت', 'spaceremit-woocommerce' ),
				'desc_tip'    => true,
			),
			'testmode'           => array(
				'title'       => __( 'وضع الاختبار', 'spaceremit-woocommerce' ),
				'label'       => __( 'تفعيل وضع الاختبار', 'spaceremit-woocommerce' ),
				'type'        => 'checkbox',
				'description' => __( 'استخدم مفاتيح API الخاصة بالاختبار', 'spaceremit-woocommerce' ),
				'default'     => 'yes',
				'desc_tip'    => true,
			),
			'test_public_key'    => array(
				'title'       => __( 'Test Public Key', 'spaceremit-woocommerce' ),
				'type'        => 'text',
				'description' => __( 'مفتاح API العام لوضع الاختبار من Spaceremit', 'spaceremit-woocommerce' ),
				'default'     => '',
				'desc_tip'    => true,
			),
			'test_secret_key'    => array(
				'title'       => __( 'Test Secret Key', 'spaceremit-woocommerce' ),
				'type'        => 'password',
				'description' => __( 'مفتاح API السري لوضع الاختبار من Spaceremit', 'spaceremit-woocommerce' ),
				'default'     => '',
				'desc_tip'    => true,
			),
			'public_key'         => array(
				'title'       => __( 'Live Public Key', 'spaceremit-woocommerce' ),
				'type'        => 'text',
				'description' => __( 'مفتاح API العام للإنتاج من Spaceremit', 'spaceremit-woocommerce' ),
				'default'     => '',
				'desc_tip'    => true,
			),
			'secret_key'         => array(
				'title'       => __( 'Live Secret Key', 'spaceremit-woocommerce' ),
				'type'        => 'password',
				'description' => __( 'مفتاح API السري للإنتاج من Spaceremit', 'spaceremit-woocommerce' ),
				'default'     => '',
				'desc_tip'    => true,
			),
			'callback_url'       => array(
				'title'       => __( 'Callback URL', 'spaceremit-woocommerce' ),
				'type'        => 'title',
				'description' => sprintf(
					'<strong>%s</strong><br>%s<br><code>%s</code>',
					__( 'استخدم هذا الرابط في إعدادات Spaceremit:', 'spaceremit-woocommerce' ),
					__( 'قم بنسخ هذا الرابط وإضافته في قسم Callback URL في إعدادات موقعك على Spaceremit', 'spaceremit-woocommerce' ),
					rest_url( 'spaceremit/v1/callback' )
				),
			),
		);
	}

	/**
	 * Process the payment and return the result.
	 *
	 * @param int $order_id Order ID.
	 * @return array
	 */
	public function process_payment( $order_id ) {
		$order = wc_get_order( $order_id );

		if ( ! $order ) {
			return array(
				'result'   => 'fail',
				'redirect' => '',
			);
		}

		// Mark as pending (we're awaiting the payment).
		$order->update_status( 'pending', __( 'في انتظار الدفع عبر Spaceremit', 'spaceremit-woocommerce' ) );

		// Store order ID in session for receipt page.
		WC()->session->set( 'spaceremit_order_id', $order_id );

		// Reduce stock levels.
		wc_reduce_stock_levels( $order_id );

		// Remove cart.
		WC()->cart->empty_cart();

		// Return thankyou redirect.
		return array(
			'result'   => 'success',
			'redirect' => $order->get_checkout_payment_url( true ),
		);
	}

	/**
	 * Output for the order received page.
	 *
	 * @param int $order_id Order ID.
	 */
	public function receipt_page( $order_id ) {
		$order = wc_get_order( $order_id );

		if ( ! $order ) {
			return;
		}

		// Make gateway instance available to template.
		$gateway = $this;

		// Include payment form template.
		include SPACEREMIT_PLUGIN_DIR . 'templates/payment-form.php';
	}

	/**
	 * Enqueue scripts.
	 */
	public function enqueue_scripts() {
		// Only load scripts on checkout-related pages
		if ( ! is_checkout() && ! is_checkout_pay_page() && ! is_order_received_page() ) {
			return;
		}

		// Get order ID from session or from URL (for receipt/payment page).
		$order_id = WC()->session->get( 'spaceremit_order_id' );
		
		// Try to get from URL if not in session (for receipt page).
		if ( ! $order_id && isset( $_GET['order'] ) && isset( $_GET['key'] ) ) {
			$order_id = absint( $_GET['order'] );
		}

		$order = $order_id ? wc_get_order( $order_id ) : null;
		
		// CRITICAL: Only load Spaceremit.js when we have an order (on receipt/payment page)
		// Don't load on checkout page where form doesn't exist yet (will cause querySelector error)
		// The form #spaceremit-form only exists in receipt_page template, not on checkout
		if ( ! $order_id || ! $order ) {
			// On checkout page without order, don't load Spaceremit.js
			// The scripts are only needed on the payment page where the form exists
			return;
		}

		// Register a script handle for inline config (loads before Spaceremit).
		wp_register_script( 'spaceremit-config', false, array( 'jquery' ), SPACEREMIT_VERSION, false );
		
		// Add inline script with configuration BEFORE Spaceremit library loads.
		$inline_config = sprintf(
			'window.SP_PUBLIC_KEY = %s;
			window.SP_FORM_ID = %s;
			window.SP_SELECT_RADIO_NAME = %s;
			window.LOCAL_METHODS_BOX_STATUS = %s;
			window.LOCAL_METHODS_PARENT_ID = %s;
			window.CARD_BOX_STATUS = %s;
			window.CARD_BOX_PARENT_ID = %s;
			window.SP_FORM_AUTO_SUBMIT_WHEN_GET_CODE = %s;',
			wp_json_encode( $this->public_key ),
			wp_json_encode( '#spaceremit-form' ),
			wp_json_encode( 'sp-pay-type-radio' ),
			'true',
			wp_json_encode( '#spaceremit-local-methods-pay' ),
			'false',
			wp_json_encode( '#spaceremit-card-pay' ),
			'true'
		);

		wp_add_inline_script( 'spaceremit-config', $inline_config );
		wp_enqueue_script( 'spaceremit-config' );

		// Enqueue Spaceremit JavaScript library (depends on config script).
		wp_enqueue_script(
			'spaceremit-js',
			'https://spaceremit.com/api/v2/js_script/spaceremit.js',
			array( 'jquery', 'spaceremit-config' ),
			SPACEREMIT_VERSION,
			true
		);

		// Enqueue our custom script.
		wp_enqueue_script(
			'spaceremit-frontend',
			SPACEREMIT_PLUGIN_URL . 'assets/js/spaceremit-frontend.js',
			array( 'jquery', 'spaceremit-js' ),
			SPACEREMIT_VERSION,
			true
		);

		// Enqueue styles.
		wp_enqueue_style(
			'spaceremit-style',
			SPACEREMIT_PLUGIN_URL . 'assets/css/spaceremit-style.css',
			array(),
			SPACEREMIT_VERSION
		);

		// Localize script with additional parameters.
		$order_key = $order ? $order->get_order_key() : '';
		
		// Get WooCommerce settings for endpoints and page IDs
		$order_received_endpoint = get_option( 'woocommerce_checkout_order_received_endpoint', 'order-received' );
		$order_pay_endpoint      = get_option( 'woocommerce_checkout_pay_endpoint', 'order-pay' );
		$checkout_page_id        = get_option( 'woocommerce_checkout_page_id', 0 );
		
		wp_localize_script(
			'spaceremit-frontend',
			'spaceremitParams',
			array(
				'publicKey'             => $this->public_key,
				'formId'                => '#spaceremit-form',
				'selectRadioName'       => 'sp-pay-type-radio',
				'localMethodsBox'       => true,
				'localMethodsParentId'  => '#spaceremit-local-methods-pay',
				'cardBox'               => false,
				'cardBoxParentId'       => '#spaceremit-card-pay',
				'autoSubmit'            => true,
				'orderId'               => $order_id,
				'orderKey'              => $order_key,
				'checkoutUrl'           => wc_get_checkout_url(),
				'checkoutPageId'        => $checkout_page_id,
				'orderReceivedEndpoint' => $order_received_endpoint,
				'orderPayEndpoint'      => $order_pay_endpoint,
				'thankYouUrl'           => $order ? $order->get_checkout_order_received_url() : wc_get_endpoint_url( 'order-received', '', wc_get_checkout_url() ),
				'orderPayUrl'           => $order ? $order->get_checkout_payment_url( true ) : wc_get_checkout_url(),
				'ajaxUrl'               => admin_url( 'admin-ajax.php' ),
				'nonce'                 => wp_create_nonce( 'spaceremit_payment_nonce' ),
				'localizedStrings'     => array(
					'paymentFailed'     => __( 'فشل الدفع. يرجى المحاولة مرة أخرى.', 'spaceremit-woocommerce' ),
					'thankYouUrlError'  => __( 'حدث خطأ أثناء تحديد رابط صفحة الاستلام.', 'spaceremit-woocommerce' ),
					'verificationError' => __( 'حدث خطأ أثناء التحقق من الدفع.', 'spaceremit-woocommerce' ),
					'serverError'       => __( 'حدث خطأ أثناء الاتصال بالخادم.', 'spaceremit-woocommerce' ),
				),
			)
		);
	}

	/**
	 * Verify payment with Spaceremit API.
	 *
	 * @param string $payment_id Payment ID from Spaceremit.
	 * @return array|WP_Error
	 */
	public function verify_payment( $payment_id ) {
		if ( empty( $payment_id ) || empty( $this->secret_key ) ) {
			return new WP_Error( 'missing_params', __( 'معاملات مفقودة للتحقق من الدفع', 'spaceremit-woocommerce' ) );
		}

		$api_url = 'https://spaceremit.com/api/v2/payment_info/';

		$response = wp_remote_post(
			$api_url,
			array(
				'timeout' => 30,
				'headers' => array(
					'Content-Type' => 'application/json',
				),
				'body'    => wp_json_encode(
					array(
						'private_key' => $this->secret_key,
						'payment_id'  => $payment_id,
					)
				),
			)
		);

		if ( is_wp_error( $response ) ) {
			return $response;
		}

		$body = wp_remote_retrieve_body( $response );
		$data = json_decode( $body, true );

		if ( ! isset( $data['response_status'] ) || 'success' !== $data['response_status'] ) {
			$message = isset( $data['message'] ) ? $data['message'] : __( 'فشل التحقق من الدفع', 'spaceremit-woocommerce' );
			return new WP_Error( 'verification_failed', $message );
		}

		return $data;
	}

	/**
	 * Check if payment status is acceptable.
	 *
	 * @param string $status_tag Status tag from Spaceremit (A, B, D, E).
	 * @return bool
	 */
	public function is_acceptable_status( $status_tag ) {
		// 'T' = Test payment (نعتبره نجاحاً في وضع الاختبار)
		$acceptable_statuses = array( 'A', 'B', 'D', 'E', 'T' );
		return in_array( $status_tag, $acceptable_statuses, true );
	}
}

