<?php
/**
 * Spaceremit Callback Handler
 *
 * @package Spaceremit_WooCommerce
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * WC_Spaceremit_Callback Class.
 */
class WC_Spaceremit_Callback {

	/**
	 * Initialize callback handler.
	 */
	public function __construct() {
		add_action( 'rest_api_init', array( $this, 'register_callback_route' ) );
	}

	/**
	 * Register REST API route for callbacks.
	 */
	public function register_callback_route() {
		register_rest_route(
			'spaceremit/v1',
			'/callback',
			array(
				'methods'             => 'POST',
				'callback'            => array( $this, 'handle_callback' ),
				'permission_callback' => '__return_true',
			)
		);
	}

	/**
	 * Handle callback from Spaceremit.
	 *
	 * @param WP_REST_Request $request REST request.
	 * @return WP_REST_Response
	 */
	public function handle_callback( $request ) {
		$body = $request->get_json_params();

		// Log callback for debugging.
		$this->log_callback( $body );

		// Validate callback data.
		if ( empty( $body['data'] ) || empty( $body['data']['id'] ) ) {
			return new WP_REST_Response(
				array(
					'success' => false,
					'message' => 'Invalid callback data',
				),
				400
			);
		}

		$payment_id = $body['data']['id'];

		// Find order by payment_id stored in meta.
		$orders = wc_get_orders(
			array(
				'meta_key'   => '_spaceremit_payment_id',
				'meta_value' => $payment_id,
				'limit'      => 1,
			)
		);

		if ( empty( $orders ) ) {
			// Try to find by notes or other methods.
			$orders = $this->find_order_by_payment_id( $payment_id );
		}

		if ( empty( $orders ) ) {
			$this->log_error( 'Order not found for payment ID: ' . $payment_id );
			return new WP_REST_Response(
				array(
					'success' => false,
					'message' => 'Order not found',
				),
				404
			);
		}

		$order = $orders[0];

		// Prevent duplicate processing
		$processed_flag = $order->get_meta( '_spaceremit_processed' );
		if ( 'yes' === $processed_flag ) {
			return new WP_REST_Response( array( 'success' => true, 'message' => 'Already processed' ), 200 );
		}
		if ( 'processing' === $processed_flag ) {
			return new WP_REST_Response( array( 'success' => true, 'message' => 'Processing' ), 200 );
		}
		$order->update_meta_data( '_spaceremit_processed', 'processing' );
		$order->save_meta_data();
		
		$gateway = new WC_Spaceremit_Gateway();

		// Verify payment with Spaceremit API.
		$verification = $gateway->verify_payment( $payment_id );

		if ( is_wp_error( $verification ) ) {
			$this->log_error( 'Payment verification failed: ' . $verification->get_error_message() );
			return new WP_REST_Response(
				array(
					'success' => false,
					'message' => 'Verification failed',
				),
				400
			);
		}

		$payment_data = $verification['data'];

		// Normalize amount: اعتمد مبلغ الطلب لتجاوز أي فروقات عمولة/رسوم.
		$order_amount   = (float) $order->get_total();
		$payment_amount = isset( $payment_data['total_amount'] ) ? (float) $payment_data['total_amount'] : $order_amount;
		$payment_data['total_amount'] = $order_amount; // نلزم استخدام مبلغ الطلب

		if ( abs( $order_amount - $payment_amount ) > 0.01 ) {
			$this->log_error( 'Amount mismatch (ignored). Order: ' . $order_amount . ', Payment: ' . $payment_amount );
		}

		$order_currency = $order->get_currency();
		$payment_currency = $payment_data['currency'];

		if ( $order_currency !== $payment_currency ) {
			$this->log_error( 'Currency mismatch. Order: ' . $order_currency . ', Payment: ' . $payment_currency );
		}

		// Update order based on payment status.
		$status_tag = $payment_data['status_tag'];
		$status = $payment_data['status'];

		// Store payment information in order meta.
		$order->update_meta_data( '_spaceremit_payment_status', $status );
		$order->update_meta_data( '_spaceremit_payment_status_tag', $status_tag );
		$order->update_meta_data( '_spaceremit_payment_data', $payment_data );
		$order->save_meta_data();

		// Update order status based on payment status.
		// Always keep order Pending for manual review.
		spaceremit_set_status_silent( $order, 'pending', false );
		spaceremit_add_note_once(
			$order,
			'spaceremit_pending',
			__( 'لديك عملية شراء جديدة قيد المراجعة، قام عميلك بارسال طلب دفع وهو بانتظار التأكيد من قبل الوكيل. الوقت المتوقع لتاكيد الطلب 0 - 12 ساعة', 'spaceremit-woocommerce' )
		);

		// Clean default WC status-change notes
		spaceremit_remove_status_notes( $order );

		$order->update_meta_data( '_spaceremit_processed', 'yes' );
		$order->save();

		return new WP_REST_Response(
			array(
				'success' => true,
				'message' => 'Callback processed successfully',
			),
			200
		);
	}

	/**
	 * Find order by payment ID using various methods.
	 *
	 * @param string $payment_id Payment ID.
	 * @return array
	 */
	private function find_order_by_payment_id( $payment_id ) {
		// Try searching by payment_code meta first (more reliable).
		$orders = wc_get_orders(
			array(
				'meta_key'   => '_spaceremit_payment_code',
				'meta_value' => $payment_id,
				'limit'      => 1,
			)
		);

		if ( ! empty( $orders ) ) {
			return $orders;
		}

		// Fallback: Search in order meta for payment_id.
		// This works with both HPOS and legacy installations.
		global $wpdb;

		// Try HPOS meta table first (if exists).
		$meta_table = $wpdb->prefix . 'wc_orders_meta';
		$order_ids = array();

		if ( $wpdb->get_var( $wpdb->prepare( "SHOW TABLES LIKE %s", $meta_table ) ) === $meta_table ) {
			// HPOS installation - use orders meta table.
			$order_ids = $wpdb->get_col(
				$wpdb->prepare(
					"SELECT DISTINCT order_id FROM {$meta_table}
					WHERE meta_key = %s
					AND meta_value = %s",
					'_spaceremit_payment_id',
					$payment_id
				)
			);
		}

		// Fallback to postmeta for legacy installations.
		if ( empty( $order_ids ) ) {
			$order_ids = $wpdb->get_col(
				$wpdb->prepare(
					"SELECT DISTINCT post_id FROM {$wpdb->postmeta}
					WHERE meta_key = %s
					AND meta_value = %s",
					'_spaceremit_payment_id',
					$payment_id
				)
			);
		}

		if ( empty( $order_ids ) ) {
			return array();
		}

		return wc_get_orders(
			array(
				'limit'   => 1,
				'include' => $order_ids,
			)
		);
	}

	/**
	 * Log callback data for debugging.
	 *
	 * @param array $data Callback data.
	 */
	private function log_callback( $data ) {
		if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
			error_log( 'Spaceremit Callback: ' . wp_json_encode( $data ) );
		}
	}

	/**
	 * Log error.
	 *
	 * @param string $message Error message.
	 */
	private function log_error( $message ) {
		if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
			error_log( 'Spaceremit Error: ' . $message );
		}
	}
}

// Initialize callback handler.
new WC_Spaceremit_Callback();

