<?php
/**
 * Spaceremit Blocks Integration
 *
 * Payment method integration for WooCommerce Blocks Checkout
 *
 * @package Spaceremit_WooCommerce
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

use Automattic\WooCommerce\Blocks\Payments\Integrations\AbstractPaymentMethodType;

/**
 * Spaceremit Blocks Integration Class.
 *
 * @extends AbstractPaymentMethodType
 */
final class Spaceremit_Blocks_Integration extends AbstractPaymentMethodType {
	/**
	 * Payment method name.
	 *
	 * @var string
	 */
	protected $name = 'spaceremit';

	/**
	 * Settings from the WP options table.
	 *
	 * @var array
	 */
	protected $settings = array();

	/**
	 * Initializes the payment method type.
	 */
	public function initialize() {
		$this->settings = get_option( 'woocommerce_spaceremit_settings', array() );
	}

	/**
	 * Returns if this payment method should be active. If false, the scripts will not be enqueued.
	 *
	 * @return boolean
	 */
	public function is_active() {
		// Check if enabled in settings (like PayPal does)
		$enabled = $this->get_setting( 'enabled', 'no' );
		$is_active = 'yes' === $enabled;
		
		if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
			error_log( 'Spaceremit Gateway: is_active() called. Enabled setting: ' . $enabled . ', Returning: ' . ( $is_active ? 'true' : 'false' ) );
		}
		
		return $is_active;
	}

	/**
	 * Returns an array of scripts/handles to be registered for this payment method.
	 *
	 * @return array
	 */
	public function get_payment_method_script_handles() {
		// Register a minimal Blocks integration script to render the payment method in Checkout block.
		wp_register_script(
			'spaceremit-blocks',
			plugins_url( 'assets/js/spaceremit-blocks.js', SPACEREMIT_PLUGIN_FILE ),
			array( 'wc-blocks-registry', 'wc-blocks-checkout', 'wp-element', 'wp-i18n' ),
			SPACEREMIT_VERSION,
			true
		);

		// Pass data to JS (title, description, supports).
		$data = array(
			'title'       => $this->get_setting( 'title', __( 'Spaceremit', 'spaceremit-woocommerce' ) ),
			'description' => $this->get_setting( 'description', '' ),
			'supports'    => $this->get_supported_features(),
		);
		wp_localize_script( 'spaceremit-blocks', 'spaceremitBlocksData', $data );

		return array( 'spaceremit-blocks' );
	}

	/**
	 * Returns an array of key=>value pairs of data made available to the payment methods script.
	 *
	 * @return array
	 */
	public function get_payment_method_data() {
		// Get gateway instance to access properties
		$gateway = null;
		if ( function_exists( 'WC' ) && WC()->payment_gateways ) {
			$gateways = WC()->payment_gateways->payment_gateways();
			if ( isset( $gateways['spaceremit'] ) ) {
				$gateway = $gateways['spaceremit'];
			}
		}

		// Get title from settings or gateway
		$title = $this->get_setting( 'title', '' );
		if ( empty( $title ) && $gateway ) {
			$title = $gateway->get_title();
		}
		if ( empty( $title ) ) {
			$title = __( 'Spaceremit', 'spaceremit-woocommerce' );
		}

		// Get description from settings or gateway
		$description = $this->get_setting( 'description', '' );
		if ( empty( $description ) && $gateway && method_exists( $gateway, 'get_description' ) ) {
			$description = $gateway->get_description();
		}

		// Get icon from gateway - format it as array like PayPal does
		$icon_html = '';
		if ( $gateway && method_exists( $gateway, 'get_icon' ) ) {
			$icon_html = $gateway->get_icon();
		}
		
		// Format icon as array (like PayPal does)
		$icon_array = array();
		if ( ! empty( $icon_html ) ) {
			// Extract icon URL from HTML if possible, or use the HTML as is
			if ( preg_match( '/<img[^>]+src=["\']([^"\']+)["\']/', $icon_html, $matches ) ) {
				$icon_array[] = array(
					'id'  => 'spaceremit',
					'alt' => 'Spaceremit',
					'src' => $matches[1],
				);
			} else {
				// If no img tag found, create a simple icon entry
				$icon_array[] = array(
					'id'  => 'spaceremit',
					'alt' => 'Spaceremit',
					'src' => '',
				);
			}
		}
		
		return array(
			'id'          => $this->name,
			'title'       => $title,
			'description' => $description,
			'supports'    => $this->get_supported_features(),
			'icon'        => $icon_array,
			// Data passed to spaceremit-blocks.js
			'blocksData'  => array(
				'title'       => $title,
				'description' => $description,
				'supports'    => $this->get_supported_features(),
			),
		);
	}

	/**
	 * Returns an array of supported features.
	 *
	 * @return array
	 */
	public function get_supported_features() {
		// Spaceremit gateway supports products by default
		return array( 'products' );
	}
}
