/**
 * Spaceremit Frontend JavaScript
 *
 * @package Spaceremit_WooCommerce
 */

(function($) {
	'use strict';

	// Initialize when DOM is ready.
	$(document).ready(function() {
		// Get parameters from localized script.
		if (typeof spaceremitParams === 'undefined') {
			console.error('Spaceremit parameters not defined');
			return;
		}

		var params = spaceremitParams;
		var localized = params.localizedStrings || {};

		// Set callback functions (must be set before Spaceremit library uses them).
		window.SP_SUCCESSFUL_PAYMENT = handlePaymentSuccess;
		window.SP_FAILD_PAYMENT = handlePaymentFail;
		window.SP_RECIVED_MESSAGE = handleMessage;
		window.SP_NEED_AUTH = handleNeedAuth;

		// Verify that global variables are set (should be set by inline script).
		if (typeof window.SP_PUBLIC_KEY === 'undefined' || typeof window.SP_FORM_ID === 'undefined') {
			console.error('Spaceremit configuration variables not found. Setting them now...');
			// Fallback: set them from params if inline script didn't work.
			window.SP_PUBLIC_KEY = params.publicKey;
			window.SP_FORM_ID = params.formId;
			window.SP_SELECT_RADIO_NAME = params.selectRadioName;
			window.LOCAL_METHODS_BOX_STATUS = params.localMethodsBox;
			window.LOCAL_METHODS_PARENT_ID = params.localMethodsParentId;
			window.CARD_BOX_STATUS = params.cardBox;
			window.CARD_BOX_PARENT_ID = params.cardBoxParentId;
			window.SP_FORM_AUTO_SUBMIT_WHEN_GET_CODE = params.autoSubmit;
		}

		// Function to ensure payment method is selected
		function ensurePaymentMethodSelected() {
			var form = document.querySelector(params.formId);
			if (!form) return;
			
			// Ensure radio button exists and is checked
			var paymentRadio = form.querySelector('input[name="sp-pay-type-radio"][value="local-methods-pay"]');
			if (paymentRadio) {
				paymentRadio.checked = true;
				// Trigger change event so Spaceremit library detects it
				var changeEvent = new Event('change', { bubbles: true });
				paymentRadio.dispatchEvent(changeEvent);
			} else {
				// Create hidden radio button if it doesn't exist
				var hiddenRadio = document.createElement('input');
				hiddenRadio.type = 'radio';
				hiddenRadio.name = 'sp-pay-type-radio';
				hiddenRadio.value = 'local-methods-pay';
				hiddenRadio.id = 'sp_local_methods_radio';
				hiddenRadio.checked = true;
				hiddenRadio.style.cssText = 'position: absolute; opacity: 0; pointer-events: none; width: 0; height: 0;';
				form.appendChild(hiddenRadio);
			}
			
			// Also set it as a data attribute on the form for Spaceremit library
			form.setAttribute('data-sp-pay-type', 'local-methods-pay');
		}
		
		// Wait for Spaceremit library to load.
		function initSpaceremit() {
			// Ensure payment method is selected (local methods is always required)
			ensurePaymentMethodSelected();
			
			// Keep checking and setting payment method periodically (in case Spaceremit resets it)
			setInterval(ensurePaymentMethodSelected, 1000);
			
			// Handle form submission (do not block Spaceremit handlers).
			$(params.formId).on('submit', function(e) {
				// Ensure payment type is set before submit
				ensurePaymentMethodSelected();
				
				// Double check just before submit - ensure radio is checked
				var paymentRadio = $(this).find('input[name="sp-pay-type-radio"][value="local-methods-pay"]');
				if (paymentRadio.length === 0) {
					// Create and append hidden radio button
					$(this).append('<input type="radio" name="sp-pay-type-radio" value="local-methods-pay" checked style="position:absolute;opacity:0;pointer-events:none;width:0;height:0;">');
				} else {
					// Ensure it's checked
					paymentRadio.prop('checked', true);
				}
				
				$('.spaceremit-loading').show();
				$('.spaceremit-error').hide();
				// let Spaceremit library handle the rest
			});
		}
		
		// Set payment method immediately on page load (before Spaceremit loads)
		ensurePaymentMethodSelected();

		// Check if Spaceremit library is already loaded.
		if (typeof window.SPACEREMIT !== 'undefined' || document.querySelector('script[src*="spaceremit.js"]')) {
			// Library is loading or loaded, wait a bit then init.
			setTimeout(initSpaceremit, 200);
		} else {
			// Wait for Spaceremit script to load.
			var checkInterval = setInterval(function() {
				if (typeof window.SPACEREMIT !== 'undefined' || document.querySelector('script[src*="spaceremit.js"]')) {
					clearInterval(checkInterval);
					setTimeout(initSpaceremit, 200);
				}
			}, 100);

			// Stop checking after 10 seconds.
			setTimeout(function() {
				clearInterval(checkInterval);
				console.error('Spaceremit library failed to load');
			}, 10000);
		}
	});

	/**
	 * Handle successful payment.
	 *
	 * @param {string} spaceremit_code Payment code from Spaceremit.
	 */
	function handlePaymentSuccess(spaceremit_code) {
		console.log('Payment successful. Code:', spaceremit_code);

		// Store payment code in form.
		$('#spaceremit-form').append(
			$('<input>').attr({
				type: 'hidden',
				name: 'SP_payment_code',
				value: spaceremit_code
			})
		);

		// Verify payment with server.
		verifyPaymentOnServer(spaceremit_code);
	}

	/**
	 * Handle failed payment.
	 */
	function handlePaymentFail() {
		console.error('Payment failed');
		$('.spaceremit-loading').hide();
		var errorMsg = localized.paymentFailed || 'Payment failed. Please try again.';
		showError(errorMsg);
	}

	/**
	 * Handle received message.
	 *
	 * @param {string} message Message from Spaceremit.
	 */
	function handleMessage(message) {
		console.log('Spaceremit message:', message);
		if (message) {
			showNotice(message);
		}
	}

	/**
	 * Handle authentication needed.
	 *
	 * @param {string} target_auth_link Authentication link.
	 */
	function handleNeedAuth(target_auth_link) {
		// console.log('Authentication needed:', target_auth_link);
		if (target_auth_link) {
			// window.location.href = target_auth_link;
		}
	}

	/**
	 * Verify payment with server.
	 *
	 * @param {string} payment_code Payment code from Spaceremit.
	 */
	function verifyPaymentOnServer(payment_code) {
		var params = spaceremitParams;

		$.ajax({
			url: params.ajaxUrl,
			type: 'POST',
			data: {
				action: 'spaceremit_verify_payment',
				payment_code: payment_code,
				order_id: params.orderId,
				nonce: params.nonce
			},
			success: function(response) {
				if (response.success) {
					// Always use thankYouUrl from params (which comes from WooCommerce)
					var thankYouUrl = params.thankYouUrl;
					
					// Fallback: construct order-received URL using WooCommerce settings if thankYouUrl is missing
					if (!thankYouUrl || thankYouUrl === '' || thankYouUrl === 'undefined') {
						console.warn('Spaceremit: thankYouUrl missing, constructing from settings');
						if (params.orderId && params.orderKey && params.checkoutUrl) {
							var checkoutUrl = params.checkoutUrl.replace(/\/$/, '');
							var orderReceivedEndpoint = params.orderReceivedEndpoint || 'order-received';
							thankYouUrl = checkoutUrl + '/' + orderReceivedEndpoint + '/' + params.orderId + '/?key=' + params.orderKey;
						}
					}
					
					if (thankYouUrl && thankYouUrl !== 'undefined') {
						console.log('Spaceremit: Redirecting to thank you page:', thankYouUrl);
						// Force immediate redirect
						window.location.replace(thankYouUrl);
					} else {
						console.error('Spaceremit: Cannot determine thank you URL', params);
						$('.spaceremit-loading').hide();
						showError(localized.thankYouUrlError || 'An error occurred while determining the thank you page URL.');
					}
				} else {
					$('.spaceremit-loading').hide();
					var fallbackMessage = localized.verificationError || 'An error occurred while verifying the payment.';
					showError(response.data?.message || fallbackMessage);
				}
			},
			error: function(xhr, status, error) {
				console.error('Spaceremit AJAX error:', status, error);
				$('.spaceremit-loading').hide();
				// Fallback: try to redirect to order-received even if AJAX fails
				// The server-side handler will catch the redirect and verify
				var thankYouUrl = params.thankYouUrl;
				if (!thankYouUrl && params.orderId && params.orderKey && params.checkoutUrl) {
					var checkoutUrl = params.checkoutUrl.replace(/\/$/, '');
					var orderReceivedEndpoint = params.orderReceivedEndpoint || 'order-received';
					thankYouUrl = checkoutUrl + '/' + orderReceivedEndpoint + '/' + params.orderId + '/?key=' + params.orderKey;
				}
				
				if (thankYouUrl && thankYouUrl !== 'undefined') {
					console.warn('Spaceremit: AJAX verification failed, redirecting anyway. Server will verify on redirect.');
					window.location.replace(thankYouUrl);
				} else {
					showError(localized.serverError || 'An error occurred while contacting the server.');
				}
			}
		});
	}

	/**
	 * Show error message.
	 *
	 * @param {string} message Error message.
	 */
	function showError(message) {
		$('.spaceremit-error .error-message').text(message);
		$('.spaceremit-error').show();
		$('.spaceremit-notice').hide();
		$('html, body').animate({
			scrollTop: $('.spaceremit-error').offset().top - 100
		}, 500);
	}

	/**
	 * Show informational message.
	 *
	 * @param {string} message Notice message.
	 */
	function showNotice(message) {
		$('.spaceremit-notice .notice-message').text(message);
		$('.spaceremit-notice').stop(true, true).fadeIn(200);
		$('html, body').animate({
			scrollTop: $('.spaceremit-notice').offset().top - 100
		}, 500);

		setTimeout(function() {
			$('.spaceremit-notice').fadeOut(200);
		}, 6000);
	}

})(jQuery);

